<?php
$pageTitle = 'Payment - Xander Global Scholars';
include 'header.php';
require_once __DIR__ . '/db.php';

// Fetch packages from database (using fee_packages table)
$packages = $conn->query("SELECT * FROM fee_packages ORDER BY title ASC");

// Fetch fee items for each package (using fee_items table)
$fee_items = [];
$fee_items_result = $conn->query("SELECT fi.*, p.title as package_title FROM fee_items fi LEFT JOIN fee_packages p ON fi.package_id = p.id ORDER BY fi.package_id, fi.id");
while ($item = $fee_items_result->fetch_assoc()) {
    if (!isset($fee_items[$item['package_id']])) {
        $fee_items[$item['package_id']] = [];
    }
    $fee_items[$item['package_id']][] = $item;
}

// Handle form submission for payment
if ($_SERVER['REQUEST_METHOD'] === 'POST' && isset($_POST['process_payment'])) {
    $package_id = (int)($_POST['package_id'] ?? 0);
    $selected_items = $_POST['selected_items'] ?? [];
    $payment_method = $_POST['payment_method'] ?? 'stripe';
    
    if ($package_id > 0 && !empty($selected_items)) {
        // Redirect to stripe payment with selected items
        $items_json = json_encode($selected_items);
        $total = 0;
        foreach ($selected_items as $item_id => $amount) {
            $total += (float)$amount;
        }
        
        header("Location: stripe-payment.php?package_id=$package_id&payment_method=$payment_method&items=" . urlencode($items_json));
        exit();
    }
}

// Calculate totals
$total_expected_sum = 0;
$total_amount_sum = 0;
$packages_summary = [];

while ($package = $packages->fetch_assoc()) {
    $packages_summary[$package['id']] = [
        'title' => $package['title'],
        'currency' => $package['currency'],
        'total_expected' => $package['total_expected'],
        'total_amount' => $package['total_amount'],
        'payment_items' => $fee_items[$package['id']] ?? []
    ];
    $total_expected_sum += $package['total_expected'];
    $total_amount_sum += $package['total_amount'];
}
?>

<div class="page-hero">
    <h1>Secure Payment Portal</h1>
    <p>Select package and fee items to complete payment securely</p>
</div>

<div class="page-content">
    <!-- Package Selection Section -->
    <div class="payment-section">
        <h2>Select Package & Fee Items</h2>
        <p>Choose a package and select fee items you want to pay for:</p>
        
        <form method="POST" id="payment-form">
            <input type="hidden" name="process_payment" value="1">
            
            <!-- Package Selection -->
            <div class="form-group">
                <label for="package-select">Select Package:</label>
                <select name="package_id" id="package-select" required onchange="loadFeeItems(this.value)">
                    <option value="">Choose a package...</option>
                    <?php foreach ($packages_summary as $package_id => $package): ?>
                    <option value="<?= $package_id ?>"><?= htmlspecialchars($package['title']) ?> - <?= $package['currency'] ?> <?= number_format($package['total_expected'], 2) ?></option>
                    <?php endforeach; ?>
                </select>
            </div>
            
            <!-- Fee Items Section -->
            <div id="fee-items-section" style="display: none;">
                <h3>Available Fee Items</h3>
                <div id="fee-items-list"></div>
                
                <!-- Total Amount -->
                <div class="total-section">
                    <h3>Total Amount: <span id="total-amount">$0.00</span></h3>
                </div>
            </div>
            
            <!-- Payment Method -->
            <div class="form-group">
                <label for="payment-method">Payment Method:</label>
                <select name="payment_method" id="payment-method" required>
                    <option value="stripe">Credit Card (Stripe)</option>
                    <option value="bank">Bank Transfer</option>
                </select>
            </div>
            
            <button type="submit" class="payment-btn" id="submit-btn" disabled>
                <i class="fas fa-credit-card"></i> Process Payment
            </button>
        </form>
    </div>
    
    <!-- Quick Actions Section -->
    <div class="payment-section">
        <h2>Payment Management</h2>
        <p>Additional payment options and tools:</p>
        
        <div class="action-grid">
            <a href="strip-test.php" class="action-card">
                <div class="action-icon">
                    <i class="fas fa-check-circle"></i>
                </div>
                <h3>Test Connection</h3>
                <p>Verify Stripe API connectivity</p>
            </a>
            
            <a href="stripe-payment.php" class="action-card">
                <div class="action-icon">
                    <i class="fas fa-credit-card"></i>
                </div>
                <h3>Quick Payment</h3>
                <p>Make a direct payment</p>
            </a>
            
            <a href="Stripe-check_transactions.php" class="action-card">
                <div class="action-icon">
                    <i class="fas fa-history"></i>
                </div>
                <h3>View Transactions</h3>
                <p>Check payment history</p>
            </a>
            
            <a href="student-manage.php" class="action-card">
                <div class="action-icon">
                    <i class="fas fa-users"></i>
                </div>
                <h3>Student Management</h3>
                <p>Manage packages and assignments</p>
            </a>
        </div>
    </div>
</div>

<?php include 'footer.php'; ?>

<script>
// Fee items data for JavaScript
const feeItemsData = <?= json_encode($fee_items) ?>;
const packagesData = <?= json_encode($packages_summary) ?>;

function loadFeeItems(packageId) {
    const feeItemsSection = document.getElementById('fee-items-section');
    const feeItemsList = document.getElementById('fee-items-list');
    const submitBtn = document.getElementById('submit-btn');
    
    if (!packageId) {
        feeItemsSection.style.display = 'none';
        submitBtn.disabled = true;
        return;
    }
    
    const items = feeItemsData[packageId] || [];
    
    if (items.length === 0) {
        feeItemsList.innerHTML = '<p>No fee items available for this package.</p>';
        feeItemsSection.style.display = 'block';
        submitBtn.disabled = true;
        return;
    }
    
    let html = '';
    
    items.forEach((item, index) => {
        const amount = parseFloat(item.total_expected || item.amount || 0);
        html += `
            <div class="fee-item">
                <div class="fee-info">
                    <h4>${item.title || item.name}</h4>
                    <small>Code: ${item.code || 'N/A'}</small>
                    ${item.refundable ? '<span class="refundable-tag">Refundable</span>' : ''}
                </div>
                <div class="fee-amount">
                    <input type="checkbox" 
                           id="fee_${item.id}" 
                           name="selected_items[${item.id}]" 
                           value="${amount}"
                           onchange="calculateTotal()">
                    <label for="fee_${item.id}">
                        ${item.currency || 'USD'} ${amount.toFixed(2)}
                    </label>
                </div>
            </div>
        `;
    });
    
    feeItemsList.innerHTML = html;
    feeItemsSection.style.display = 'block';
    calculateTotal();
}

function calculateTotal() {
    const checkboxes = document.querySelectorAll('input[name^="selected_items"]:checked');
    let total = 0;
    
    checkboxes.forEach(checkbox => {
        total += parseFloat(checkbox.value) || 0;
    });
    
    document.getElementById('total-amount').textContent = `$${total.toFixed(2)}`;
    document.getElementById('submit-btn').disabled = total === 0;
}
</script>

<style>
.payment-section {
    background: var(--card);
    border-radius: 15px;
    padding: 30px;
    margin-bottom: 30px;
    box-shadow: var(--shadow-md);
    border: 1px solid var(--border);
}

.form-group {
    margin-bottom: 25px;
}

.form-group label {
    display: block;
    margin-bottom: 8px;
    font-weight: 600;
    color: var(--text);
}

.form-group select {
    width: 100%;
    padding: 12px;
    border: 2px solid var(--border);
    border-radius: 8px;
    font-size: 1rem;
    background: white;
}

.fee-item {
    display: flex;
    justify-content: space-between;
    align-items: center;
    padding: 15px;
    border: 1px solid var(--border);
    border-radius: 8px;
    margin-bottom: 10px;
    background: var(--bg);
}

.fee-info h4 {
    margin: 0 0 5px 0;
    color: var(--text);
}

.fee-info small {
    color: var(--text-light);
}

.refundable-tag {
    background: #10b981;
    color: white;
    padding: 2px 8px;
    border-radius: 12px;
    font-size: 0.75rem;
    margin-left: 10px;
}

.fee-amount {
    display: flex;
    align-items: center;
    gap: 10px;
}

.fee-amount input[type="checkbox"] {
    width: 20px;
    height: 20px;
    cursor: pointer;
}

.fee-amount label {
    font-weight: 600;
    color: var(--primary);
    cursor: pointer;
}

.total-section {
    background: linear-gradient(135deg, var(--primary) 0%, var(--primary-dark) 100%);
    color: white;
    padding: 20px;
    border-radius: 10px;
    text-align: center;
    margin-top: 20px;
}

.total-section h3 {
    margin: 0;
    font-size: 1.5rem;
}

.payment-btn {
    width: 100%;
    background: linear-gradient(135deg, var(--accent) 0%, var(--accent-dark) 100%);
    color: white;
    border: none;
    padding: 18px 24px;
    border-radius: 12px;
    font-weight: 600;
    font-size: 1.1rem;
    cursor: pointer;
    transition: var(--transition);
    margin-top: 20px;
    display: flex;
    align-items: center;
    justify-content: center;
    gap: 10px;
}

.payment-btn:hover:not(:disabled) {
    transform: translateY(-2px);
    box-shadow: 0 6px 16px rgba(255, 140, 66, 0.3);
}

.payment-btn:disabled {
    opacity: 0.6;
    cursor: not-allowed;
}

.action-grid {
    display: grid;
    grid-template-columns: repeat(auto-fit, minmax(250px, 1fr));
    gap: 20px;
    margin-top: 20px;
}

.action-card {
    background: var(--bg);
    border: 1px solid var(--border);
    border-radius: 12px;
    padding: 20px;
    text-decoration: none;
    color: var(--text);
    transition: var(--transition);
    display: block;
}

.action-card:hover {
    transform: translateY(-2px);
    box-shadow: var(--shadow-md);
    border-color: var(--accent);
}

.action-icon {
    width: 50px;
    height: 50px;
    background: var(--accent);
    border-radius: 50%;
    display: flex;
    align-items: center;
    justify-content: center;
    color: white;
    font-size: 1.2rem;
    margin-bottom: 15px;
}

.action-card h3 {
    margin: 0 0 8px 0;
    color: var(--text);
}

.action-card p {
    margin: 0;
    color: var(--text-light);
    font-size: 0.9rem;
}
</style>
